function P = permsk(S, K, ~)
%PERMSK  All permutations of K elements from a set
%
%   P = PERMSK(S, K) returns all permutations of K elements taken from the
%   set S with N elements. P will be a N!/(N-K)!-by-K array. Each row of P
%   holds a unique permutation of K elements.
%   P will be of the same type as S, which can be any kind of array
%   (numerical, char, string, struct, cell, etc.). 
%
%   Example:
%      permsk(1:4, 2) % -> [ 1 2 ; 1 3 ; 1 4 ; 2 1 ; ... 
%                            ... 3 2 ; 3 4 ; 4 1 ; 4 2 ; 4 3]
%
%   The list of permutations in the rows of P are ordered. To speed things
%   up slightly, you can provide a third argument, like: PERMSK(S, K, 1),
%   so a call to sortrows is skipped. When K > N, P will be empty.
%
%   See also PERMS, NCHOOSEK, RANDPERM, PERMUTE.
%            PERMN, NCHOOSE, NCHOOSE2 (File Exchange)

% version 1.1 (apr 2019)
% Matlab File Exchange Author ID: 10584
% email: samelinoa@gmail.com

% 1.0 (apr 2019) - created
% 1.1 (apr 2019) - fixed some typos, extended help and comments

narginchk(2, 3) ;
% any 3rd argument will cause skipping of a call to sortrows

if K == 1
    P = S(:) ; % each row holds a single element of S
else
    % S has N elements. Use NCHOOSEK to retrieve a list of all ordered
    % permutations of K indices from the set 1:N.
    I = nchoosek(1:numel(S), K) ;

    % Use PERMS to get all permutations of this list of indices.
    % I use K:-1:1 to get the result in somewhat the right sequence, which
    % makes sortrows(I) possibly a little faster.
    I = I(:, perms(K:-1:1)) ;
    % reshape to return a N!/(N-K)!-by-K index array
    I = reshape(I , [] , K) ;

    if nargin == 2  
        % any third argument would skip this call
        I = sortrows(I) ;
    end

    % index into S to retrieve the permutations
    P = S(I) ;
end
